#!/bin/bash

# Set global variables:
#   - servers
#   - redundant_conf 
#   - valid_redundant_conf
#   - remote_server_indeces
#   - my_ip_addr
#   - my_id
#   - pm
#   - el
#   - fifo_dir
#   - tail_filter
#   - logfile
#
# Fix the owner/group setting of /opt/passlogic/data/conf/redundant.conf
# to passlogic:passlogic 
# (restore.sh modifies it to apache:apache.)
#
pm_init() {
    local ip_com line i

    redundant_conf=/opt/passlogic/data/conf/redundant.conf
    ip_com=`/sbin/ip -f inet addr`

    [ ! -e $redundant_conf ] && touch $redundant_conf
    chown passlogic:passlogic $redundant_conf

    servers=()
    for line in `sed -e 's/SERVER[0123]=//' $redundant_conf`; do
	servers=("${servers[@]}" $line)
	echo $ip_com | grep -q "${line}/" && my_ip_addr=$line
    done

    [ -z $my_ip_addr ] && my_ip_addr='localhost'

    [ ${#servers[@]} -lt 2 ] && valid_redundant_conf=false || valid_redundant_conf=true

    remote_server_indeces=()
    for i in $(seq 0 $((${#servers[@]}-1))); do
	if [ ! $my_ip_addr = ${servers[$i]} ]
	then
	    remote_server_indeces=("${remote_server_indeces[@]}" $i)
	else
	    my_id=$i
	fi
    done

    pm=/opt/passlogic/apps/tools/pm.sh
    el=`sed 's/.*release\s\([0-9]\).*/\1/' /etc/redhat-release`
    fifo_dir=/home/passlogic

    if [ -z "$verbose" ]
    then
	tail_filter="tail -1"
	exec 2>/dev/null
    else
	tail_filter="cat"
    fi

    logfile=/var/log/passlogic/pm.log
}

pipestatus() {
    local ps=("${PIPESTATUS[@]}")
    local argv=("$@")
    local ret=0
    local i

    for ((i=0; i<$#; ++i))
    do
	[ ! ${ps[$i]} -eq ${argv[$i]} ] && ret=1 && break
    done
    return $ret
}

show_activity() { echo "`LANG=C date "+%Y-%m-%d %H:%M:%S"`: $my_ip_addr: $*" | tee -a $logfile ; }
log_activity() { echo "`LANG=C date "+%Y-%m-%d %H:%M:%S"`: $my_ip_addr: $*" >> $logfile ; }

show_ok_ng() { [ $? -eq 0 ] && echo :ok || echo :ng; }

service_() {
    case $el in
	"6")
	    /sbin/service $1 $2
	    ;;
	"7")
	    /bin/systemctl $2 $1
	    ;;
    esac
}

# name of PostgreSQL process in an output of 'ps -upasslogic -o comm'
#
postmaster() {
    case $el in
	"6")
	    echo postmaster
	    ;;
	"7")
	    echo postgres
	    ;;
    esac
}

# read the conf value from settings.conf
#   ex) read_settings_conf PL_SESSION_TIMEOUT
#
read_settings_conf() {
    if [ -e /opt/passlogic/data/conf/settings.conf ]
    then
	sed -n "s/^$1\s*=\s*\(\S\+\)\s*$/\1/p" /opt/passlogic/data/conf/settings.conf | head -1
    fi
}

wait_socket_5439() { while : ; do netstat -tnlp 2> /dev/null | grep `postmaster` | grep -q 5439 && break; done; echo :ok;}

wait_pgsql_up() {
    while :
    do
	/opt/passlogic/pgsql/bin/psql -h localhost -p 5439 -d passlogic -U passlogic -t -c "select 'ready'" | grep -q "ready" && break
    done
    echo :ok
}

wait_pgpool_up() {
    while :
    do
	/opt/passlogic/pgsql/bin/psql -h localhost -p 9915 -d passlogic -U passlogic -t -c "select 'ready'" | grep -q "ready" && break
    done
    echo :ok
}

wait_socket_9915_9925() {
    while :
    do
	netstat -tnlp 2>/dev/null | grep pgpool | grep -q 9915 &&
	    netstat -tnlp 2>/dev/null | grep pgpool | grep -q 9925 && break
    done
    echo :ok
}

wait_postmaster_disappear() { while : ; do ps -upasslogic -o comm | grep -q -E `postmaster` || break; done; echo :ok; }

wait_pgpool_disappear() { while : ; do ps -upasslogic -o comm | grep -q pgpool || break; done; echo :ok; }

wait_httpd_disappear() { while : ; do ps -e -o comm | grep -q -E '^httpd$' || break; done; echo :ok; }

ping() {
    echo "pong from $my_ip_addr"
    echo :ok
}

pgsql_down() {
    service_ passlogic-pgsql stop
    timeout 10 $pm <<< "wait_postmaster_disappear"
    [ $? -eq 124 ] && echo :ng || echo :ok
}

pgsql_up_check() {
    local ret
    timeout 10 $pm <<< "wait_socket_5439"
    if [ $? -eq 124 ]
    then
	ret=:ng
    else
	timeout 10 $pm <<< "wait_pgsql_up"
	[ $? -eq 124 ] && ret=:ng || ret=:ok
    fi
    echo $ret
}

pgsql_up() { service_ passlogic-pgsql start; show_ok_ng; }

pgsql_status() {
    timeout 10 /opt/passlogic/pgsql/bin/psql -p 5439 -d passlogic -U passlogic -t -c "select 'ready'" | grep -q "ready"
    pipestatus 0 0 && echo :ok || echo :ng
}

pgpool_down() {
    service_ passlogic-pgpool stop
    timeout 10 $pm <<< "wait_pgpool_disappear"
    [ $? -eq 124 ] && echo :ng || echo :ok
}

pgpool_up() {
    local ret
    service_ passlogic-pgpool start
    timeout 10 $pm <<< "wait_socket_9915_9925"
    if [ $? -eq 124 ]
    then
	ret=:ng
    else
	timeout 10 $pm <<< "wait_pgpool_up"
	[ $? -eq 124 ] && ret=:ng || ret=:ok
    fi
    echo $ret
}

pgpool_status() {
    timeout 10 /opt/passlogic/pgsql/bin/psql -p 9915 -d passlogic -U passlogic -t -c "select 'ready'" | grep -q "ready"
    pipestatus 0 0 && echo :ok || echo :ng
}

node_info() {
    local node_info
    node_info=`/opt/passlogic/pgpool/bin/pcp_node_info 10 localhost 9925 passlogic passlogic $1`
    if [ $? -eq 0 ]
    then
	echo $node_info | sed 's/.*\s5439\s\([0-9]\).*/\1/'
    else
	echo :ng
    fi
}

detach_node() {
    /opt/passlogic/pgpool/bin/pcp_detach_node 10 localhost 9925 passlogic passlogic $1
}

attach_node() {
    /opt/passlogic/pgpool/bin/pcp_attach_node 10 localhost 9925 passlogic passlogic $1
}

detach_remote() {
    local status ret
    ret=:ok
    for i in ${remote_server_indeces[@]}
    do
	[ `node_info $i` = "3" ] || detach_node $i
	[ `node_info $i` = ":ng" ] && ret=:ng && break
    done
    echo $ret
}

attach_remote() {
    local status ret
    ret=:ok
    for i in ${remote_server_indeces[@]}
    do
	attach_node $i
	grep -q -E '^[12]' <<< `node_info $i`
	[ ! $? -eq 0 ] && ret=:ng && break
    done
    echo $ret
}

httpd_down() {
    service_ httpd stop
    timeout 10 $pm <<< "wait_httpd_disappear"
    [ $? -eq 124 ] && echo :ng || echo :ok
}

httpd_up() { service_ httpd start; show_ok_ng; }

httpd_graceful() {
    local com
    case $el in
	"6")
	    com=graceful
	    ;;
	"7")
	    com=reload
	    ;;
    esac
    service_ httpd $com
    show_ok_ng;
}

httpd_status() { service_ httpd status; show_ok_ng; }

radiusd_restart() { service_ radiusd restart; show_ok_ng; }

# Copy following directories;
#   - /opt/passlogic/pgsql/archives
#   - /opt/passlogic/pgsql/data
#
# which contains files generated by passlogic_redundant.sh
#
#   - /opt/passlogic/pgsql/data/pg_hba.conf
#   - /opt/passlogic/pgsql/data/failover_mail.sh
#   - /opt/passlogic/pgsql/data/failback_mail.sh
#
copy_pgsql_archives_data_dirs() {
    local i rsync_com
    for i in ${remote_server_indeces[@]}
    do
	echo copying to ${servers[$i]} ...
	rsync_com=$(cat << EOS
        rsync -v -c -az --delete -e ssh \
              --exclude postmaster.opts \
              --exclude postmaster.pid \
              --exclude serverlog \
              --exclude bin \
              --exclude include \
              --exclude lib \
              --exclude share \
              /opt/passlogic/pgsql/ ${servers[$i]}:/opt/passlogic/pgsql/
EOS
)
	echo $rsync_com
	su - passlogic -c "$rsync_com"
	show_ok_ng
    done
}

# Copy following files generated by passlogic_redundant.sh
#
#   - /opt/passlogic/pgpool/etc/pgpool.conf
#   - /opt/passlogic/data/conf/redundant.conf
#
copy_pgpool_redundant_conf() {
    local i scp_com
    for i in ${remote_server_indeces[@]}
    do
	scp_com=$(cat << EOS
        scp -p /opt/passlogic/pgpool/etc/pgpool.conf ${servers[$i]}:/opt/passlogic/pgpool/etc/ &&
        scp -p /opt/passlogic/data/conf/redundant.conf ${servers[$i]}:/opt/passlogic/data/conf/
EOS
)
	echo $scp_com
	su - passlogic -c "$scp_com"
	show_ok_ng
    done
}

remove_cache() { rm -f /opt/passlogic/tmp/*.cache; show_ok_ng; }

cleanup_session_table() {
    local ret lifetime cond_t

    lifetime=`read_settings_conf PL_SESSION_TIMEOUT`
    grep -q -E '^[0-9]+$' <<< "$lifetime"
    if [ $? -eq 0 ]
    then
	cond_t=$(( `date +%s` - $lifetime ))
	/opt/passlogic/pgsql/bin/psql -h localhost -p 5439 -d passlogic -U passlogic -t <<EOF
	delete from passlogic_session where updated_at < $cond_t
EOF
	[ $? -eq 0 ] && ret=:ok || ret=:ng
    else
	ret=:ng
    fi
    echo $ret
}

transfer_settings() {
    local i rsync_com
    for i in ${remote_server_indeces[@]}
    do
	echo copying to ${servers[$i]} ...
	rsync_com=$(cat << EOS
        rsync -a -e ssh --delete /home/passlogic/exported_settings/${my_id} ${servers[$i]}:/home/passlogic/exported_settings/
EOS
)
	echo $rsync_com
	su - passlogic -c "$rsync_com"
	show_ok_ng
    done
}

export_settings() {
    su - passlogic -c "mkdir -m 755 -p /home/passlogic/exported_settings/"
    su - passlogic -c "mkdir -m 755 -p /home/passlogic/exported_settings/${my_id}"
    # Remark on rsync options
    # o --chmod=Fo=r makes exported passlogic*.cron readable by passlogic user.
    # o --exclude="*" inhibits expoting files other than passlogic*.cron and passchg_mail*
    rsync -a --chmod=Fo=r --include="passlogic*" --include="passchg_mail*" --exclude="*" --delete \
	  /etc/cron.d/ /home/passlogic/exported_settings/${my_id}/cron.d/ && \
	rsync -a --exclude redundant.conf --delete /opt/passlogic/data/ /home/passlogic/exported_settings/$my_id/data/ && \
	chown -R passlogic:passlogic  /home/passlogic/exported_settings/
    show_ok_ng
}

import_cron.d() {
    local master_node_id=$1
    # Remark on rsync option
    # o -I causes all files to be updated
    rsync -rt -I --chmod=Fo-r --include="passlogic*" --include="passchg_mail*" --exclude="*" --delete \
	  /home/passlogic/exported_settings/${master_node_id}/cron.d/ /etc/cron.d/
    show_ok_ng
}

import_conf() {
    local master_node_id=$1
    # Remark on the rsync option
    # o -I ignore times causing all files to be updated
    # o PKI certs are imported by import_cert
    rsync -rt -I --exclude cert/ \
	  /home/passlogic/exported_settings/${master_node_id}/data/conf/ /opt/passlogic/data/conf/ && \
    chown apache:apache \
	  /opt/passlogic/data/conf \
	  /opt/passlogic/data/conf/clients.conf \
	  /opt/passlogic/data/conf/passlogic-admin.conf \
	  /opt/passlogic/data/conf/passlogic-config.xml \
	  /opt/passlogic/data/conf/passlogic-ui-lang.xml \
	  /opt/passlogic/data/conf/xauth_passlogic_00.conf \
	  /opt/passlogic/data/conf/xauth_passlogic_10.conf \
	  /opt/passlogic/data/conf/xauth_passlogic_20.conf \
	  /opt/passlogic/data/conf/settings.conf \
	  /opt/passlogic/data/conf/outerapi.conf && \
    find /opt/passlogic/data/conf/ -type f -name "saml-*" -exec chown apache:apache {} \; && \
    fsync logo_images.png apache:apache
    show_ok_ng
}

import_cert() {
    local master_node_id=$1
    # Remark on the rsync option
    # o -I ignore times causing all files to be updated
    rsync -rt -I --delete /home/passlogic/exported_settings/${master_node_id}/data/conf/cert/ /opt/passlogic/data/conf/cert/ && \
    chown apache:apache /opt/passlogic/data/conf/cert/*
    show_ok_ng
}

import_license() {
    local master_node_id=$1
    cp -f --preserve=timestamps /home/passlogic/exported_settings/${master_node_id}/data/license-ent.asc /opt/passlogic/data/ && \
	chown apache:apache /opt/passlogic/data/license-ent.asc
    show_ok_ng
}

lock() {
	local pid=$1
	local lockfile_basename=$2
	local lockfile=/opt/passlogic/data/lock/$2
	local interval=$3
	local process_monitor=${@:4}
	local ret
	
	mkdir -p /opt/passlogic/data/lock
	chown passlogic:passlogic /opt/passlogic/data/lock
	chmod 755 /opt/passlogic/data/lock
	su - passlogic -c "flock $lockfile bash -c \"echo -n $pid > $lockfile && chmod 444 $lockfile\""
	ret=$?

	nohup $pm <<< "lock_col $pid $lockfile_basename $interval $process_monitor" > /dev/null 2>&1 &

	[ $ret -eq 0 ]
	show_ok_ng
}

unlock() {
	local pid=$1
	local lockfile=/opt/passlogic/data/lock/$2
	local ret=0
	
	if [ -r $lockfile ] && [ `cat $lockfile` == "$pid" ]
	then
		rm $lockfile
		ret=0
	else
		ret=1
	fi

	[ $ret -eq 0 ]
	show_ok_ng
}

lock_col() {
	local pid=$1
	local lockfile=/opt//passlogic/data/lock/$2
	local lockfile_basename=$2
	local interval=$3
	local process_monitor=${@:4}
	local ret=0

	# PIPE -> KILL を止める
	trap '' SIGPIPE
	# TERM -> ロックファイル削除
	trap 'rm -f $lockfile; log_activity lock_col receieved SIGTERM. lockfile removed' SIGTERM
	while :
	do
		[ ! -r $lockfile ] && break
		[ `cat $lockfile` != "$pid" ] && ret=1 && break
		$process_monitor
		if [ $? -eq 1 ]
		then
			if [ -r $lockfile  ]
			then
				log_activity lock_col unlockd $lockfile_basename
				$pm <<< "unlock $pid $lockfile_basename"
				break
			fi
		fi
		sleep $interval
	done
	[ $ret -eq 0 ]
	show_ok_ng
}

# proc_mon
#
#   引数を=で区切って
#   １番目をps, ２番目をgrep の引数として扱う
# 
proc_mon() {
	local ps_opt
	local grep_opt

	ps_opt=`echo $* | cut -f 1 -d'='`
	grep_opt=`echo $* | cut -f 2 -d'='`

	ps ${ps_opt} | grep -v grep | grep -q ${grep_opt}
	show_ok_ng
}

replication?() {
	$valid_redundant_conf
	show_ok_ng
}

# o sync a file in exported_settings/${master_node_id}/data/conf/ with /opt/passlogic/data/conf/
# o change a file owner/group to a specified owner/group
fsync() {
    local file=$1
    local owner_group=$2
    ( ( [ ! -e /home/passlogic/exported_settings/${master_node_id}/data/${file} ] && rm -f /opt/passlogic/data/${file} ) || \
      ( cp -f --preserve=timestamps /home/passlogic/exported_settings/${master_node_id}/data/${file} /opt/passlogic/data/${file} && \
	      chown ${owner_group} /opt/passlogic/data/${file}) )
}

send() {
    local argv=("$@")
    local host=${servers[${argv[0]}]}
    echo "${argv[@]:1}" | su - passlogic -c "ssh -T $host sudo $pm"
}

call_internal_func() {
    $* 2>&1 | tee -a $logfile | tail -1 | grep -q -E '^:ok$'; show_ok_ng
}

call_remote_func() {
    local line
    for i in ${remote_server_indeces[@]}; do
	send $i $* 2>&1 | tee -a $logfile | while read line
	do
	    grep -q -E '^:ok$' <<< "$line"
	done
	show_ok_ng
    done
}

show_error_message() { echo $* | tee -a $logfile; echo :ng | tee -a $logfile; }

while getopts ":v" OPT
do
    case $OPT in
	v)
	    verbose=0
	    ;;
    esac
done

pm_init

while read msg
do
    ary=(`echo $msg | tr -s ' '`)
    c=${ary[0]}
    opts=("${ary[@]:1}")
    show_activity $c "${opts[@]}"
    case $c in
	ping|pgsql_*|pgpool_*|httpd_*|radiusd_restart|remove_cache|cleanup_session_table|wait_*|lock|lock_col|proc_mon|unlock|log_activity|replication?)
	    call_internal_func $c "${opts[@]}"
	    ;;
	copy_*|export_settings|transfer_settings|import_*|??tach_remote|wait_*)
	    if $valid_redundant_conf
	    then
		call_internal_func $c "${opts[@]}"
	    else
		show_error_message "Error in calling $c: please confirm the servers configuration in $redundant_conf"
	    fi
	    ;;
	remote_*)
	    if $valid_redundant_conf
	    then
		remote_com=${c/remote_/}
		call_remote_func $remote_com "${opts[@]}"
	    else
		show_error_message "Error in calling $c: please confirm the servers configuration in $redundant_conf"
	    fi
	    ;;
	deploy_remote_conf|deploy_remote_cron.d|deploy_remote_license|deploy_remote_cert)
	    if $valid_redundant_conf
	    then
		remote_com=import_${c/deploy_remote_/}
		call_remote_func $remote_com $my_id
	    else
		show_error_message "Error in calling $c: please confirm the servers configuration in $redundant_conf"
	    fi
	    ;;
    esac
done | $tail_filter
