<?php

/**
 * restore.sh が作ったlockfile がある場合は exit 1
 */
require_once('/opt/passlogic/apps/lib/passlogic_lock.php');
if (PasslogicLock::cron_exists())
{
	exit(1);
}

date_default_timezone_set("Asia/Tokyo");

define("BASE_DIR", dirname(dirname(__FILE__)));
define("LIB_DIR", BASE_DIR . '/lib');
define("PEAR_DIR", LIB_DIR . '/pear');
define("DATA_DIR", '/opt/passlogic/data');
define("TMP_DIR", "/opt/passlogic/tmp");

require_once(LIB_DIR."/config_reader.inc.php");
require_once(LIB_DIR."/constants.inc.php");
require_once(LIB_DIR."/language.inc.php");
require_once(LIB_DIR."/log.inc.php");
require_once(LIB_DIR."/crypt.inc.php");
require_once(LIB_DIR."/ldap.inc.php");
require_once(LIB_DIR."/xmlconf.inc.php");
require_once(LIB_DIR."/database/databaseManager.inc.php");

$settings = new ConfigReader(DATA_DIR.'/conf/settings.conf');
$settings->set_global_define();

$Language = new Language;


class ldapsync {

	private $domain = 'local';
	private $uid    = array();
	private $uemail = array();
	
	private $plLog  = NULL;
	
	private $ascii_default_mapping = array(
		'uid'     => 'ldapAttrUid',
		'policy'  => 'ldapAttrPolicy',
		'group1'  => 'ldapAttrGroup1',
		'group2'  => 'ldapAttrGroup2',
		'group3'  => 'ldapAttrGroup3',
		'group4'  => 'ldapAttrGroup4',
		'group5'  => 'ldapAttrGroup5',
		'uemail'  => 'ldapAttrUemail',
		'uexpiry' => 'ldapAttrUexpiry',
	);
	private $multibyte_default_mapping = array(
		'uname'           => 'ldapAttrUname',
		'employee_number' => 'ldapAttrEmployee_number',
		'section'         => 'ldapAttrSection',
		'phone'           => 'ldapAttrPhone',
		'ucomment'        => 'ldapAttrUcomment',
		'attribute1'      => 'ldapAttrAttribute1',
		'attribute2'      => 'ldapAttrAttribute2',
		'attribute3'      => 'ldapAttrAttribute3',
		'attribute4'      => 'ldapAttrAttribute4',
		'attribute5'      => 'ldapAttrAttribute5',
		'attribute6'      => 'ldapAttrAttribute6',
		'attribute7'      => 'ldapAttrAttribute7',
		'attribute8'      => 'ldapAttrAttribute8',
		'attribute9'      => 'ldapAttrAttribute9',
		'attribute10'     => 'ldapAttrAttribute10',
		'sslvpnsso1'      => 'ldapAttrSslvpnsso1',
		'sslvpnsso2'      => 'ldapAttrSslvpnsso2',
		'sslvpnsso3'      => 'ldapAttrSslvpnsso3',
		'sslvpnsso4'      => 'ldapAttrSslvpnsso4',
		'sslvpnsso5'      => 'ldapAttrSslvpnsso5',
		'sslvpnsso6'      => 'ldapAttrSslvpnsso6',
		'sslvpnsso7'      => 'ldapAttrSslvpnsso7',
		'sslvpnsso8'      => 'ldapAttrSslvpnsso8',
		'sslvpnsso9'      => 'ldapAttrSslvpnsso9',
		'sslvpnsso10'     => 'ldapAttrSslvpnsso10',
		'param1'          => 'ldapAttrParam1',
		'param2'          => 'ldapAttrParam2',
		'param3'          => 'ldapAttrParam3',
		'param4'          => 'ldapAttrParam4',
		'param5'          => 'ldapAttrParam5',
		'param6'          => 'ldapAttrParam6',
		'param7'          => 'ldapAttrParam7',
		'param8'          => 'ldapAttrParam8',
		'param9'          => 'ldapAttrParam9',
		'param10'         => 'ldapAttrParam10',
	);

	/**
	 * LDAP ID同期に設定されたドメインを読み込んで、同期対象のドメインに対応した設定を返す
	 * 
	 * @param string $target_domain 設定を読み込む対象のドメイン
	 * @return $ldap_config 該当ドメインに対応した設定の連想配列(無ければ、NULLを戻す)
	 */
	private function read_ldap_config($target_domain) {
		$ldap_config = NULL;

		// LDAP設定取得
		$xml_confobj = new xmlconf();
		$xml_conf    = $xml_confobj->_get_xmlconf();
		$xml_ldap_conf = $xml_conf['syncLdaps']['ldapConfig'];

		$syncLDAP_list = array();
		// 設定が複数ある場合は、配列を上書き
		if ( $xml_ldap_conf and $xml_ldap_conf[0] ) {
			$syncLDAP_list = $xml_ldap_conf;
		}
		// 設定が一つしかない場合は、配列に追加
		else {
			$syncLDAP_list[] = $xml_ldap_conf;
		}
		
		// LDAP ID同期ドメインの一覧から該当ドメインのものを読み込み
		if ( $syncLDAP_list and is_array($syncLDAP_list) ) {
			foreach ( $syncLDAP_list as $syncLDAP ) {
				if ( $syncLDAP['ldapDomain'] == $target_domain ) {
					$ldap_config = $syncLDAP;
					break;
				}
			}
		}

		return $ldap_config;
	}

	/**
	 * 
	 * コンストラクタ
	 * 
	 */
	function __construct() {

		$this->plLog = new PasslogicLog();
	}

	/**
	 * LDAPサーバとの接続処理
	 * 
	 * @param string $server_name 接続先のLDAPサーバのホスト名(もしくはIPアドレス)
	 * @param array $ldap_config LDAP接続設定の連想配列
	 * @return 成功の場合は LDAP BINDオブジェクト、失敗の場合は NULL
	 */
	private function bind_LDAP_server($server_name, $ldap_config) {
		global $settings;
		
		$protocol = ( $ldap_config['ldapSecure'] ? 'ldaps' : 'ldap' );
		$port     = $ldap_config['ldapPort'];
		
		// 接続用URIの構文チェック(実際の接続はこの時点では未実施)
		$ldap_connection = ldap_connect("{$protocol}://{$server_name}:{$port}");
		if ( $errnum = ldap_errno($ldap_connection) ) {
			$errstr = ldap_err2str($errnum);
			$this->plLog->log( "31213", '', $this->domain, array('additional_message'=>"{$server_name} : {$errstr}") );
			return NULL;
		}

		// LDAPサーバ接続用の各種パラメータ設定
		$timeout = $settings->get_int_value_or_default("LDAP_NETWORK_TIMEOUT", LDAP_DEFAULT_NETWORK_TIMEOUT);
		ldap_set_option($ldap_connection, LDAP_OPT_NETWORK_TIMEOUT, $timeout);

		if ( ! ldap_set_option($ldap_connection, LDAP_OPT_PROTOCOL_VERSION, 3) ) {
			$errnum = ldap_errno($ldap_connection);
			$errstr = ldap_err2str($errnum);
			$this->plLog->log( "31214", '', $this->domain, array('additional_message'=>"{$server_name} : {$errstr}") );
			return NULL;
		}

		if ( $ldap_config['ldapType'] == 'ActiveDirectory' ) {
			ldap_get_option($ldap_connection, LDAP_OPT_REFERRALS, $ret_params);
			if ( $ret_params ) {
				if( ! ldap_set_option($ldap_connection, LDAP_OPT_REFERRALS, 0)){
					$errnum = ldap_errno($ldap_connection);
					$errstr = ldap_err2str($errnum);
					$this->plLog->log( "31215", '', $this->domain, array('additional_message'=>"{$server_name} : {$errstr}") );
					return NULL;
				}
			}
		}
		
		// LDAPサーバ接続
		@ldap_bind($ldap_connection, $ldap_config['ldapBinddn'], $ldap_config['ldapBindpasswd']);
		if ( $errnum = ldap_errno($ldap_connection) ) {
			$errstr = ldap_err2str($errnum);
			$this->plLog->log( "31216", '', $this->domain, array('additional_message'=>"{$server_name} : {$errstr}") );
			return NULL;
		}
		
		return $ldap_connection;
	}

	/**
	 * 外部から呼び出すメイン実行処理部分
	 * 
	 * @param string $targetLdapDomain 同期対象のドメイン名
	 * @param boolean $auto_execute 
	 * @return
	 */
	public function main($targetLdapDomain, $auto_execute) {
		
		$ldap_config = $this->read_ldap_config($targetLdapDomain);
		if ( $ldap_config['ldapDomain'] ) {
			$this->domain = $ldap_config['ldapDomain'];
		}

		// 処理開始(31201: Start exporting user data to PassLogic from LDAP.)
		$this->plLog->log( "31201", '', $this->domain );

		if ( !$targetLdapDomain ) {
			$plLog->log( "31211", '', $this->domain );
			exit;
		}

		if ( $ldap_config and is_array($ldap_config) ) {
			$crypt = new Crypt();
			$ldap_config['ldapBindpasswd'] = $crypt->decrypt_string($ldap_config['ldapBindpasswd'], $ldap_config['ldapDomain']);
		}
		else {
			$this->plLog->log( "31212", '', $this->domain );
			exit;
		}

		// 対象のLDAPサーバに接続
		$ldap_servers = array(
			$ldap_config['ldapHostname'],
			$ldap_config['ldapHostname2'],
		);
		$ldap_connection = NULL;
		foreach ( $ldap_servers as $ldap_server ) {
			if ( !$ldap_server ) {
				continue;
			}
			if ( $ldap_connection = $this->bind_LDAP_server($ldap_server, $ldap_config) ) {
				break;
			}
		}
		if ( !$ldap_connection ) {
			$this->plLog->log( "31216", '', $this->domain, array('additional_message'=>'no LDAP server can be binded.') );
			exit;
		}

		// LDAPサーバで検索する際、取得対象の属性名の一覧を作成
		// - ${ascii,multibyte}_default_mappingのうち、属性名の指定があるものを
		//   設定ファイルから取得してきて、データベース列名とLDAP属性名の新しいマッピング連想配列を作成
		$search_attributes = array();
		$ascii_mapping = array();
		foreach ( $this->ascii_default_mapping as $column_name => $attribute_name ) {
			if ( $ldap_config[$attribute_name] ) {
				$search_attributes[] = $ldap_config[$attribute_name];
				$ascii_mapping[$column_name] = strtolower($ldap_config[$attribute_name]);
			}
		}
		$multibyte_mapping = array();
		foreach ( $this->multibyte_default_mapping as $column_name => $attribute_name ) {
			if ( $ldap_config[$attribute_name] ) {
				$search_attributes[] = $ldap_config[$attribute_name];
				$multibyte_mapping[$column_name] = strtolower($ldap_config[$attribute_name]);
			}
		}
		// 別々のデータベース列名から、同一のLDAP属性名を参照する場合があるので注意
		$search_attributes = array_values(array_unique($search_attributes));

		// LDAP検索処理
		$ldapsearch = ldap_search($ldap_connection, $ldap_config['ldapTopObject'], $ldap_config['ldapFilter'], $search_attributes);
		if ( $errnum = ldap_errno($ldap_connection) ) {
			$errstr = ldap_err2str($errnum);
			$this->plLog->log( "31217", '', $this->domain, array('additional_message'=>$errstr) );
			exit;
		}

		$ldap_entries = ldap_get_entries($ldap_connection, $ldapsearch);
		if ( $errnum = ldap_errno($ldap_connection) ) {
			$errstr = ldap_err2str($errnum);
			$this->plLog->log( "31218", '', $this->domain, array('additional_message'=>$errstr) );
			exit;
		}

		// LDAP検索結果をもとに、PassLogicデータベースに格納用のユーザ情報の配列を作成
		$ldap_users = array();
		$num_entries = $ldap_entries['count'];
		for ($i = 0; $i < $num_entries; $i++) {
			foreach ( $ldap_entries[$i] as $n => $v) {
				$name = strtolower($n);
				$value = NULL;

				switch ( strtolower($name) ) {
					case 'dn':
						$value = $v;
						break;
					case 'objectguid':
						$value = base64_encode($v[0]);
						break;
					default:
						$value = $v[0];
				}
				
				if ( !$ldap_users[$i] ) {
					$ldap_users[$i] = array();
				}
				foreach ($ascii_mapping as $column_name => $attribute_name ) {
					if ( $attribute_name == $name ) {
						$ldap_users[$i][$column_name] = $value;
					}
				}
				foreach ( $multibyte_mapping as $column_name => $attribute_name ) {
					if ( $attribute_name == $name ) {
						$ldap_users[$i][$column_name] = mb_convert_encoding($value, "sjis-win", "UTF-8");
					}
				}
			}
		}

		// 完全同期(削除込みでの同期)の場合、生存フラグ用の列を初期化
		$user_db = NULL;
		if ( $ldap_config['ldapRunMode'] == 'sync' ) {
			$user_db = new passlogic_user(getPDO());
			if( $user_db->query_userlist(array('domain' => $this->domain), array('uid')) ) {
				$user_db->initialize_syncLDAP_flag($this->domain);
			}
		}

		// 取得したLDAPユーザデータをもとにCSVファイルを作成
		$csv_data = '';
		$uid_list = array();
		$column_names = array_merge(array_keys($ascii_mapping), array_keys($multibyte_mapping));
		// ポリシーは必須になるので、なければ追加
		if ( !in_array('policy', $column_names) ) {
			$column_names[] = 'policy';
		}
		foreach ( $ldap_users as $ldap_user ) {
			// 先頭の要素は、削除フラグ、uid、domainで固定
			$csv_data .= '"",';
			$csv_data .= '"' . preg_replace("/\"/", "\"\"", $ldap_user['uid'] ) .'",';
			$csv_data .= '"' . $this->domain . '",';
			// その後は列順に従って要素を配置
			foreach ( $column_names as $column_name ) {
				switch ( $column_name ) {
					case 'uid':
						break;
					case 'policy':
						$policy0   = ( $ldap_user[$column_name] ? $ldap_user[$column_name] : $ldap_config['ldapDefaultPolicy'] );
						$csv_data .= '"'. $policy0 .'",';
						break;
					case 'expire':
						$csv_data .= '"' . $ldap_user[$column_name] . '",';
						break;
					default:
						$csv_data .= '"' . preg_replace("/\"/", "\"\"", $ldap_user[$column_name]) . '",';
				}
			}
			$csv_data  = preg_replace('/\,$/', '', $csv_data);
			$csv_data .= "\n";

			// 完全同期(削除込みでの同期)の場合、存在フラグを設定
			if ( $ldap_config['ldapRunMode'] == 'sync' ) {
				if ( $user_db->exist_userdata($ldap_user['uid'], $this->domain) ) {
					$uid_list[] = $ldap_user['uid'];
				}
			}
		}

		// 完全同期(削除込みでの同期)の場合、存在フラグのないユーザの削除用CSVを追加
		if ( $ldap_config['ldapRunMode'] == 'sync' ) {
			$user_db->set_syncLDAP_flag($uid_list, $this->domain);
			$no_users = $user_db->get_syncLDAP_nonExistUsers($this->domain);
			if ( $no_users ) {
				// 同期対象外ユーザを削除フラグを立ててcsvに追加(削除はcsvの行頭に書込み)
				foreach ( $no_users as $no_user ) {
					$csv_data = ( '"d","' . $no_user['uid'] . '","' . $this->domain . '"' . "\n" . $csv_data );
				}
			}
		}

		$uniqid = uniqid();
		$tmpfile_name = TMP_DIR."/userLdapCmd_csv_".$uniqid;
		file_put_contents($tmpfile_name, $csv_data);

		// CSVインポート用の設定ファイルを作成
		$csv_config  = "0\n";
		$csv_config .= ( $ldap_config['ldapAddactMail'] ? '1' : '0' )."\n";
		$csv_config .= "delflag,uid,domain,";
		foreach ( $column_names as $column_name ) {
			switch ( $column_name ) {
				case 'uid':
					break;
				default:
					$csv_config .= "{$column_name},";
			}
		}
		$csv_config = preg_replace('/\,$/', '', $csv_config);

		$config_file = TMP_DIR."/userLdapCmd_configfile_".$uniqid;
		file_put_contents($config_file, $csv_config);

		$this->plLog->log( "31202", '', $this->domain );

		// 最後にuserimportコマンドを実行して、作成したCSVファイルをもとにユーザデータ更新
		$cmd = '/usr/bin/php /opt/passlogic/apps/admin/cgi/userimport.php ';
		$cmd .= escapeshellarg($tmpfile_name).' ';
		$cmd .= escapeshellarg($config_file)." '' '' '' ";
		if ($auto_execute == 1) {
			$cmd .= escapeshellarg($this->domain);
		}
		$cmd .= ' > /dev/null &';
		$void = exec($cmd, $ret);
	}
}

$targetLdapDomain = $argv[1];
$auto_execute = $argv[2];
$ldapsync = new ldapsync();
$ldapsync->main($targetLdapDomain, $auto_execute);

?>
