#!/usr/bin/php
<?php
/**
 * restore.sh が作ったlockfile がある場合は exit 1
 */
require_once('/opt/passlogic/apps/lib/passlogic_lock.php');
if (PasslogicLock::cron_exists())
{
	exit(1);
}

// application DEBUG true / false
define('DEBUG', 0);

date_default_timezone_set('Asia/Tokyo');

/**
 * An absolute file-system path to the lib directory.
 */
define('PRE_BASE_DIR', '/opt/passlogic');
define('BASE_DIR', PRE_BASE_DIR . '/apps');

/**
 * constants define
 */
define('TMP_DIR', PRE_BASE_DIR . '/tmp');
define('LIB_DIR', BASE_DIR . '/lib');
define('PEAR_DIR', LIB_DIR . '/pear');
define('DATA_DIR', PRE_BASE_DIR . '/data');
define('CONF_DIR', DATA_DIR . '/conf');
// クライアント証明書発行時にcert_utilで必要
define('CERT_DIR', CONF_DIR . '/cert');
define('TOOLS_DIR', BASE_DIR . '/tools');

// setting.confのdefineの読み込み
require_once(LIB_DIR . "/config_reader.inc.php");
$settings = new ConfigReader(CONF_DIR . '/settings.conf');
$settings->set_global_define();

// 必要なクラスの読み込み
require_once(LIB_DIR . "/log.inc.php");
require_once(LIB_DIR . "/xmlconf.inc.php");
require_once(LIB_DIR . "/validator.inc.php");
require_once(LIB_DIR . "/passlogic.inc.php");
require_once(LIB_DIR . "/passlogic_api_util.inc.php");
require_once(LIB_DIR . '/model/user_model.php');
require_once(LIB_DIR . '/crypt.inc.php');
require_once(LIB_DIR . '/cert_util.inc.php');

class userimport {

	public function __construct() {
		$this->validator = new Validator();
		$this->plinc = new passlogicInc();
		$this->plau = new passlogic_api_util();
		$this->cert = new cert_util();
		$this->plLog = $this->plau->plLog;
		$this->xmlconf = $this->plau->xmlconf;
		$this->crypt = $this->plau->crypt;
	}

	public function main($import_file_path, $config_file_path, $host_ip, $client_ip, $client_ua, $domain_name) {
		// ログ用の出力パラメータ
		$this->client_info = array(
		    'host_ip' => $host_ip,
		    'client_ip' => $client_ip,
		    'client_ua' => $client_ua
		);
		$this->plau->set_logoptions($this->client_info);

		// インポート前の設定
		$checked = $this->pre_check($import_file_path, $config_file_path, $domain_name);
		if ($checked) {
			// プレチェック失敗=>インポートファイルとコンフィグファイルを削除
			// import_file delete.
			if (is_file($import_file_path)) {
				@unlink($import_file_path);
			}
			// config_file delete.
			if (is_file($config_file_path)) {
				@unlink($config_file_path);
			}
			$this->plLog->log($checked['code'], '', $domain_name, $this->client_info);
			return false;
		}

		// 設定ファイル読み込み
		$this->import_config($config_file_path);

		// インポート開始のログ出力
		$this->plLog->log('31301', '', $domain_name, $this->client_info);

		// インポートのCSVファイル読み込み
		$this->import_csv($import_file_path);

		// インポート終了のログ出力
		$this->plLog->log('31302', '', $domain_name, $this->client_info);

		// import_file delete.
		if (is_file($import_file_path)) {
			@unlink($import_file_path);
		}
		// config_file delete.
		if (is_file($config_file_path)) {
			@unlink($config_file_path);
		}
		//ロックファイル回収
		unlink($this->lock_file);
		return true;
	}

	/**
	 * 読込み前のチェック
	 * @param type $import_file_path
	 * @param type $config_file_path
	 * @param type $domain_name
	 * @return エラーコード、もしくはNULL
	 */
	private function pre_check($import_file_path, $config_file_path, $domain_name = NULL) {
		//入力ファイル確認
		if (!is_readable($import_file_path)) {
			return array('code' => '31304');
		}
		if (!is_readable($config_file_path)) {
			return array('code' => '31305');
		}

		// ロックファイル確認
		$lock_file_base = TMP_DIR . "/userimport.php.lock";
		$this->lock_file = $domain_name ? $lock_file_base . "." . $domain_name : $lock_file_base;

		// userimport.php.lock*ではじまるすべてのロックファイルを検査
		foreach (glob($lock_file_base . "*") as $file) {
			if (file_exists($file)) {
				//5分以上処理が継続していなければロックファイル削除
				if ((time() - filemtime($file)) > 60 * 5) {
					unlink($file);
				} else {
					// ドメイン未指定のuserimport	: いずれのuserimportが処理中であっても中断
					// ドメイン指定のuserimport	: 同一ドメイン または ドメイン未指定のuserimportが処理中のときだけ中断
					if (!$domain_name || $file == $this->lock_file || $file == $lock_file_base) {
						return array('code' => '31303');
					}
				}
			}
		}

		// 多重起動防止用ロックファイル作成
		touch($this->lock_file);

		// 問題ないのでNULLを返す
		return NULL;
	}

	/**
	 * コンフィグファイルを読込む
	 */
	private function import_config($config_file_path) {
		// from passlogic-admin.
		$fp = fopen($config_file_path, 'rb');
		$this->skip_itemname = rtrim(fgets($fp, 4089));
		$second_line = rtrim(fgets($fp, 4089));
		$this->notify_by_email = $second_line & 1;
		$this->basic_auth_param = $second_line & 2;
		$this->set_column = preg_split("/,/", rtrim(fgets($fp, 4089)));
		fclose($fp);
	}

	private function import_csv($import_file_path) {
		// インポートファイルの操作オブジェクト
		$fp = new SplFileObject($import_file_path);

		// 先頭行を項目名としてスキップ
		if ($this->skip_itemname) {
			$fp->fgets();
		}

		// 1行ずつ読み込み
		while (!$fp->eof()) {
			// 1行目読み込み
			// PHPのCSV取り込みは、SJISの『表』『圭』などの文字を取り込めないので、CSV変換前にUTF-8化する
			$line = mb_convert_encoding($fp->fgets(), "UTF-8", "sjis-win, UTF-8, eucjp-win, auto");
			if (!$line) {
				break;
			}
			$arr = str_getcsv($line);
			// データが配列じゃない（取り込み失敗）
			if (!is_array($arr)) {
				break;
			}

			// CSVのデータを並べ替え
			$param = NULL;
			$column_index = 0;
			foreach ($arr as $val) {
				if ($this->set_column[$column_index]) {
					$param[$this->set_column[$column_index]] = $val;
				}
				$column_index++;
			}
			$this->import_user($param);
		}
	}

	private function import_user($param) {
		if (preg_match("/^d$/", $param['delflag'])) {
			// 削除フラグのユーザを削除
			$user = new user_model($param['uid'], $param['domain']);
			$ret = $this->plau->userdelete($param['uid'], $param['domain']);
			if ($ret['code'] != '30003') {
				// 正常に追加できていない場合、そこで処理終了
				return false;
			}
			return true;
		} else {
			// ユーザの新規・編集
			// ユーザパラメータのValidation
			if (!$this->check_user_param($param)) {
				// パラメータのフォーマットに問題あり
				return false;
			}

			// Basic認証のparam1-10はBase64＋暗号化保存
			if ($this->basic_auth_param) {
				for ($i = 1; $i <= 10; $i++) {
					if ($param['param' . $i]) {
						$base64_passwd = base64_encode($param['param' . $i]);
						$param['param' . $i] = $this->crypt->encrypt_string($base64_passwd, $param['uid'] . $param['domain']);
					}
				}
			}

			// ロック状態で登録する場合は現在のタイムスタンプも登録
			if ($param['locked'] == 1) {
				$param['lockedtime'] = $_SERVER['REQUEST_TIME'];
			}

			// ユーザの作成
			$user = new user_model($param['uid'], $param['domain']);

			// ハードウェアトークンのシリアル番号重複をチェック
			if (array_key_exists('token_serial', $param) && ($param['token_serial'] || $param['token_serial'] === '0')) {
				if ($user->check_token($param['token_serial'])) {
					$this->plLog->log("31012", $param['uid'], $param['domain'], $this->client_info);
					return false;
				}
			}
			// 次ハードウェアトークンのシリアル番号重複をチェック
			if (array_key_exists('next_token_serial', $param) && ($param['next_token_serial'] || $param['next_token_serial'] === '0')) {
				if ($user->check_token($param['next_token_serial'])) {
					$this->plLog->log("31012", $param['uid'], $param['domain'], $this->client_info);
					return false;
				}
			}
			// ハードウェアトークンのシリアル番号と次ハードウェアトークンのシリアル番号が重複する場合をチェック
			if (array_key_exists('token_serial', $param) && array_key_exists('next_token_serial', $param)) {
				if ($param['token_serial'] === $param['next_token_serial'] && $param['token_serial'] !== '') {
					$this->plLog->log("31012", $param['uid'], $param['domain'], $this->client_info);
					return false;
				}
			}

			// ユーザの有無チェック
			if (!$user->is_exist()) {
				// 新規ユーザ
				// パラメータの詰込み
				foreach ($param as $key => $val) {
					$user->set_value($key, $val);
				}
				$user->set_value('lastauthdate', 0);

				// ポリシー取得
				$user->set_config();
				$config = $user->get_config();
				$auth_method = $config['authMethod'];

				// 認証方式ごとのパラメータセット
				if ($auth_method == 'PassClip') {
					// PassClip
					$sendnotice_tmpl_type = 'passclip';
					// シード作成
					$user->set_secret();
					// ハードウェアトークン解除
					$user->del_value('token_serial');
					$user->del_value('next_token_serial');
				} elseif ($auth_method == 'TOTP') {
					// TOTP
					$sendnotice_tmpl_type = 'totp';
					// 必須パラメータ(token_serial)が無ければ、エラー
					if (!(array_key_exists('token_serial', $param) && ($param['token_serial'] || $param['token_serial'] === '0'))) {
						$this->plLog->log("31012", $param['uid'], $param['domain'], $this->client_info);
						return false;
					}
				} else {
					// PassLogic
					$sendnotice_tmpl_type = 'new';

					// csvにシークレットパターンのデータ項目あり
					if (array_key_exists('secret_pattern', $param)) {
						// 指定されている値をcreaternd関数用にコピー
						$initialPattern = $param['secret_pattern'];
					}
					// csvにスタティックパスワードのデータ項目あり
					if (array_key_exists('static_password', $param)) {
						// 指定されている値をcreaternd関数用にコピー
						$initialPin = $param['static_password'];
					}

					// どちらも未定義の場合は、シークレットパターンのみランダム発行
					if (!isset($initialPattern) and ! isset($initialPin)) {
						$initialPattern = 'random';
					}
					// シークレットパターンとスタティックパスワードの組合わせがランダム発行可能かチェック
					// なお、チェック用メソッド用に、判定対象の文字列を変換
					if ($initialPattern == 'random') {
						$secret_pattern0 = '__RANDOM_ASSIGN__';
					} else {
						$secret_pattern0 = $initialPattern;
					}
					if (Util::checkRandomCombination($secret_pattern0, $config['randomOtpLength'], NULL, $initialPin, $config['randomPinLength'])) {
						$this->plLog->log("31012", $param['uid'], $param['domain'], $this->client_info);
						return false;
					}

					// シークレットパターン設定
					$user->set_p1($initialPattern, $initialPin);
					// ハードウェアトークン解除
					$user->del_value('token_serial');
					$user->del_value('next_token_serial');
				}

				// ユーザデータをDBに書き込み
				$ret = $this->plau->usercreate($user);
				if ($ret['code'] != '30001') {
					// 正常に追加できていない場合、そこで処理終了
					return false;
				}

				if ($ret && $this->notify_by_email) {
					$tmp_params = array(
					    "uid" => $param['uid'],
					    "domain" => $param['domain'],
					    "type" => "user",
					    "num" => null,
					    "client_info" => $this->client_info
					);
					$this->plinc->sendnotice($sendnotice_tmpl_type, $tmp_params);
				}

				// PKI自動発行ポリシーの場合
				if ($config['pkicheck'] && $config['autopki'] && $config['autopki_timing'] != '1') {
					// 証明書自動発行
					$CL_set = $this->cert->generate_cert($param['uid'], $param['domain'], $param['uemail'], $config['autopki_CNSetting'], $config['autopki_expireDate']);
					if ($CL_set && $this->notify_by_email) {
						// PKI発行メール送信
						$tmp_params['serial'] = $CL_set['serial'];
						$this->plinc->sendnotice('pki', $tmp_params);
					}
				}
			} else {
				// ユーザ既存情報取り込み
				$user->select_from_db();

				// 編集内容の有無フラグ
				$edit = false;
				// 編集パラメータの詰込み
				foreach ($param as $key => $val) {
					if ($user->set_value($key, $val, true) === true) {
						$edit = true;
					}
				}
				$user->set_value('lastauthdate', 0);

				// ユーザ更新
				if ($edit) {
					$ret = $this->plau->useredit($user);
					if ($ret['code'] != '30002') {
						// 正常に追加できていない場合、そこで処理終了
						return false;
					}
				}
			}

			//ロックファイル更新
			touch($this->lock_file);
		}
	}

// 未整理 -------------------------------------------------------------//

	/**
	 * validator main
	 */
	private function _validator($valid_arr, $param) {
		if (!is_array($param)) {
			return false;
		}

		foreach ($valid_arr as $name => $valid) {
			if (!is_array($valid)) {
				return false;
			}
			foreach ($valid as $val) {
				$rule = $val['rule'];
				$ret = $this->validator->dispatchMethod($param["$name"], $rule);

				if (!$ret[0]) {
					if (is_array($rule)) {
						$params['error'] = $name . ":" . $rule[0];
					} else {
						$params['error'] = $name . ":" . $rule;
					}
					$params['code'] = $val['code'];
					break;
				}
			}
		}
		return $params;
	}

	private function check_user_param($param) {
		$valid_arr = array(
		    'uid' => array(
			'Required' => array(
			    'rule' => 'Required',
			    'code' => '31012',
			),
			'Between' => array(
			    'rule' => array('Between', 1, 30),
			    'code' => '31012',
			),
			'Userid' => array(
			    'rule' => 'Userid',
			    'code' => '31012',
			),
		    ),
		    'domain' => array(
			'Required' => array(
			    'rule' => 'Required',
			    'code' => '31012',
			),
			'Between' => array(
			    'rule' => array('Between', 1, 30),
			    'code' => '31012',
			),
			'Domain' => array(
			    'rule' => 'Domain',
			    'code' => '31012',
			),
		    ),
		    'uemail' => array(
			'Email' => array(
			    'rule' => 'Email',
			    'code' => '31012',
			),
		    ),
		    'policy' => array(
			'Policy' => array(
			    'rule' => 'Policy',
			    'code' => '31012',
			),
		    ),
		    'group1' => array(
			'Group' => array(
			    'rule' => 'Group',
			    'code' => '31012',
			),
		    ),
		    'group2' => array(
			'Group' => array(
			    'rule' => 'Group',
			    'code' => '31012',
			),
		    ),
		    'group3' => array(
			'Group' => array(
			    'rule' => 'Group',
			    'code' => '31012',
			),
		    ),
		    'group4' => array(
			'Group' => array(
			    'rule' => 'Group',
			    'code' => '31012',
			),
		    ),
		    'group5' => array(
			'Group' => array(
			    'rule' => 'Group',
			    'code' => '31012',
			),
		    ),
		    'udisabled' => array(
			'Regexp' => array(
			    'rule' => array('Regexp', "01"),
			    'code' => '31012',
			),
		    ),
		    'uexpiry' => array(
			'Date' => array(
			    'rule' => 'Date',
			    'code' => '31012',
			),
		    ),
		    'static_password' => array(
			'Fixpassword' => array(
			    'rule' => 'Fixpassword',
			    'code' => '31012',
			),
		    ),
		    'token_serial' => array(
			'PrintableString' => array(
			    'rule' => 'PrintableString',
			    'code' => '31012',
			),
			'Between' => array(
			    'rule' => array('Between', 0, 32),
			    'code' => '31012',
			),
		    ),
		    'next_token_serial' => array(
			'PrintableString' => array(
			    'rule' => 'PrintableString',
			    'code' => '31012',
			),
			'Between' => array(
			    'rule' => array('Between', 0, 32),
			    'code' => '31012',
			),
		    ),
		    'token_pin' => array(
			'Fixpassword' => array(
			    'rule' => 'Fixpassword',
			    'code' => '31012',
			),
			'Between' => array(
			    'rule' => array('Between', 0, 255),
			    'code' => '31012',
			),
		    ),
		);

		// validator
		$error = $this->_validator($valid_arr, $param);
		if ($error) {
			$this->plLog->log($error['code'], $param['uid'], $param['domain'], $this->client_info);
			return false;
		}

		// ドメイン名存在チェック
		if ($param['domain'] != "local") {
			$groupnames = array();
			if (isset($this->xmlconf['passlogicConfig']['ldap']['ldapConfig'])) {
				if (isset($this->xmlconf['passlogicConfig']['ldap']['ldapConfig'][0])) {
					foreach ($this->xmlconf['passlogicConfig']['ldap']['ldapConfig'] as $val) {
						$groupnames[] = $val["groupname"];
					}
				} else {
					$groupnames[] = $this->xmlconf['passlogicConfig']['ldap']['ldapConfig']["groupname"];
				}
			}
			if (!in_array($param['domain'], $groupnames)) {
				// 存在しなかった
				$this->plLog->log("31012", $param['uid'], $param['domain'], $this->client_info);
				return false;
			}
		}

		// ポリシー存在チェック
		if ($param['policy']) {
			if (is_array($this->xmlconf['policies']['data'])) {
				$policies = $this->xmlconf['policies']['data'];
			} else {
				$policies[] = $this->xmlconf['policies']['data'];
			}
			if (!in_array($param['policy'], $policies)) {
				// 存在しなかった
				$this->plLog->log("31012", $param['uid'], $param['domain'], $this->client_info);
				return false;
			}
		}

		// シークレットパターンチェック
		if ($param['initialPattern']) {
			if ($param['initialPattern'] !== 'random') {
				if (preg_match("/[^0-9,]+/", $param['initialPattern']) || preg_match("/[,]{2,}/", $param['initialPattern'])) {
					$this->plLog->log("31012", $param['uid'], $param['domain'], $this->client_info);
					return false;
				}
			}
		}

		return true;
	}

}

/* --- 引数定義 --------------------------- */
// $argv[1] import file path
// $argv[2] config file path
// $argv[3] server host ip address
// $argv[4] client ip address
// $argv[5] client user agent
// $argv[6] domain name (LDAP ID同期実行時のドメイン指定)
//   LDAP ID同期の場合、$argv[3]～$argv[5]は空文字列になる('')、
//   $client_ip, $client_uaは、inet型なので空文字列だとログ書込み時にエラーが出るので、
//   NULLで初期化(合わせて、$argv[5]もNULLで初期化)
$import_file_path = $argv[1];
$config_file_path = $argv[2];
$host_ip = ( $argv[3] ? $argv[3] : NULL );
$client_ip = ( $argv[4] ? $argv[4] : NULL );
$client_ua = ( $argv[5] ? $argv[5] : NULL );
$domain_name = $argv[6] ? $argv[6] : '';

$userimport = new userimport();
$userimport->main($import_file_path, $config_file_path, $host_ip, $client_ip, $client_ua, $domain_name);

